/* $Id: gate3.c,v 1.28 1999/02/18 00:42:50 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Thoroughly test data enabling functions */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"
#include "err1432.h"

#define	MAG_SQUARED

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024

#ifdef	MAG_SQUARED
#define FBLOCKSIZE	(BLOCKSIZE / 2)
#else
#define	FBLOCKSIZE	BLOCKSIZE
#endif

#define	MODE_MISC	0
#define	MODE_ONOFF	1
#define	MODE_SCAN	2
#define	MODE_SLEEP	3
#define	MODE_MAX	3

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: gate3.c,v 1.28 1999/02/18 00:42:50 ericb Exp $";
static const char *progname;

static void
my_usleep(unsigned int usec)
{
    struct timeval timeout;

    timeout.tv_sec = usec / 1000000;
    timeout.tv_usec = usec % 1000000;
    (void) select(0, 0, 0, 0, &timeout);
}

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int data_mode, int fft)
{
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_data_mode(hw, group, data_mode));
    if (data_mode == E1432_CONTINUOUS_MODE)
	CHECK(e1432_set_span(hw, group, 4000));
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    if (fft)
    {
	CHECK(e1432_set_calc_data(hw, group, E1432_DATA_FREQ));
#ifdef	MAG_SQUARED
	CHECK(e1432_set_avg_mode(hw, group, E1432_AVG_RMS));
	CHECK(e1432_set_avg_number(hw, group, 1));
#endif
    }
    CHECK(e1432_init_measure(hw, group));

    return 0;
}

static int
change_enable_misc(E1432ID hw, int group, int nchan,
		   SHORTSIZ16 *chan_list, int fft, int
		   enabled[][NCHAN_MAX])
{
    static int count;
    int     type, chan, status;

    count++;
    switch (count % 17)
    {
    case 0:
	/* Enable all channels */
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
	    {
		if (type > 0 && !fft)
		    enabled[type][chan] = 0;
		else
		    enabled[type][chan] = 1;
	    }
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_ON));
	break;
    case 1:
	/* Do nothing */
	break;
    case 2:
	/* Do nothing */
	break;
    case 3:
    case 4:
	/* Disable all channels, then re-enable all */
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_OFF));

	if ((count % 5) == 0)
	{
	    status = e1432_block_available(hw, group);
	    if (status != 0)
	    {
		(void) fprintf(stderr,
			       "%s: e1432_block_available: returned"
			       " %d with no enabled channels\n",
			       progname, status);
		return -1;
	    }
	}
	else
	    my_usleep(1000);

	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_ON));
	break;
    case 5:
	/* Disable all, then enable only a time channel */
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
		enabled[type][chan] = 0;
	enabled[0][0] = 1;
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_OFF));
	CHECK(e1432_set_enable(hw, chan_list[0],
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	break;
    case 6:
	/* Rotate time channel, disabling in between */
	enabled[0][0] = 0;
	enabled[0][3] = 1;
	CHECK(e1432_set_enable(hw, chan_list[0],
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	my_usleep(100);
	CHECK(e1432_set_enable(hw, chan_list[3],
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	break;
    case 7:
	/* Do nothing */
	break;
    case 8:
	/* Rotate time channel, not disabling in between */
	enabled[0][3] = 0;
	enabled[0][2] = 1;
	CHECK(e1432_set_enable(hw, chan_list[2],
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	CHECK(e1432_set_enable(hw, chan_list[3],
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	break;
    case 9:
	/* Add time channel */
	enabled[0][nchan - 1] = 1;
	CHECK(e1432_set_enable(hw, chan_list[nchan - 1],
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	break;
    case 10:
	/* Do nothing */
	break;
    case 11:
	/* Disable all, then enable only a freq channel (if fft enabled) */
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
		enabled[type][chan] = 0;
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_OFF));
	enabled[fft][nchan / 2] = 1;
	CHECK(e1432_set_enable(hw, chan_list[nchan / 2],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	break;
    case 12:
	/* Do nothing */
	break;
    case 13:
	/* Rotate channel, disabling in between */
	enabled[fft][nchan / 2] = 0;
	enabled[fft][0] = 1;
	CHECK(e1432_set_enable(hw, chan_list[nchan / 2],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	my_usleep(100);
	CHECK(e1432_set_enable(hw, chan_list[0],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	break;
    case 14:
	/* Rotate channel, disabling in between */
	if (fft)
	{
	    enabled[1][0] = 0;
	    enabled[1][1] = 1;
	    CHECK(e1432_set_enable(hw, chan_list[0],
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_OFF));
	    CHECK(e1432_set_enable(hw, chan_list[1],
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_ON));
	}
	else
	{
	    enabled[0][0] = 0;
	    enabled[0][1] = 1;
	    CHECK(e1432_set_enable(hw, chan_list[0],
				   E1432_ENABLE_TYPE_TIME,
				   E1432_ENABLE_OFF));
	    CHECK(e1432_set_enable(hw, chan_list[1],
				   E1432_ENABLE_TYPE_TIME,
				   E1432_ENABLE_ON));
	}
	break;
    case 15:
	/* Add channel */
	if (fft)
	{
	    enabled[1][nchan - 1] = 1;
	    CHECK(e1432_set_enable(hw, chan_list[nchan - 1],
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_ON));
	}
	else
	{
	    enabled[0][nchan - 1] = 1;
	    CHECK(e1432_set_enable(hw, chan_list[nchan - 1],
				   E1432_ENABLE_TYPE_TIME,
				   E1432_ENABLE_ON));
	}
	break;
    case 16:
	/* Do nothing */
	break;
    }

    return 0;
}

static int
change_enable_onoff(E1432ID hw, int group, int nchan,
		    SHORTSIZ16 *chan_list, int fft,
		    int enabled[][NCHAN_MAX])
{
    static int count, echan;
    int     type, chan, next, status;

    if (count == 0)
    {
	/* Disable all but one channel */
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
		enabled[type][chan] = 0;
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_OFF));

	echan = 0;
	enabled[fft][echan] = 1;
	CHECK(e1432_set_enable(hw, chan_list[0],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
    }

    count++;
    if ((count % 2) == 1)
    {
	/* Cycle through all channels, disabling the current before
	   enabling the next channel. */
	next = echan - 1;
	if (next < 0)
	    next = nchan - 1;

	enabled[fft][echan] = 0;
	CHECK(e1432_set_enable(hw, chan_list[echan],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));

	if ((count % 5) == 0)
	{
	    status = e1432_block_available(hw, group);
	    if (status != 0)
	    {
		(void) fprintf(stderr,
			       "%s: e1432_block_available: returned"
			       " %d with no enabled channels\n",
			       progname, status);
		return -1;
	    }
	}
	else
	    my_usleep(100);

	enabled[fft][next] = 1;
	CHECK(e1432_set_enable(hw, chan_list[next],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	echan = next;
    }

    return 0;
}

static int
change_enable_scan(E1432ID hw, int group, int nchan,
		   SHORTSIZ16 *chan_list, int fft,
		   int enabled[][NCHAN_MAX])
{
    static int count, echan;
    int     type, chan, next;

    if (count == 0)
    {
	/* Disable all but one channel */
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
		enabled[type][chan] = 0;
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_OFF));

	echan = 0;
	enabled[fft][echan] = 1;
	CHECK(e1432_set_enable(hw, chan_list[0],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
    }

    count++;
    if ((count % 3) == 0)
    {
	/* Cycle through all channels, enabling the next before
	   disabling the current channel. */
	next = echan - 1;
	if (next < 0)
	    next = nchan - 1;

	enabled[fft][echan] = 0;
	enabled[fft][next] = 1;
	CHECK(e1432_set_enable(hw, chan_list[next],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	CHECK(e1432_set_enable(hw, chan_list[echan],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	echan = next;
    }

    return 0;
}

static int
change_enable_sleep(E1432ID hw, int group, int nchan,
		    SHORTSIZ16 *chan_list, int fft,
		    int enabled[][NCHAN_MAX], int verbose)
{
    static int count;
    int     type, chan;

    /* By sleeping after changing the enable settings, we can
       sometimes find a bug in multi-module handling. */

    switch (count++ % 32)
    {
    case 8:
	/* Enable only time data on channel 0 */
	if (verbose > 0)
	    (void) printf("Enable time on chan 0\n");
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
		enabled[type][chan] = 0;
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_FREQ,
			       E1432_ENABLE_OFF));
	CHECK(e1432_set_enable(hw, chan_list[0],
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	enabled[0][0] = 1;

	(void) sleep(1);
	break;
    case 16:
	/* Enable all data */
	if (verbose > 0)
	    (void) printf("Enable all\n");
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
		if (type > 0 && !fft)
		    enabled[type][chan] = 0;
		else
		    enabled[type][chan] = 1;
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	if (fft)
	    CHECK(e1432_set_enable(hw, group,
				   E1432_ENABLE_TYPE_FREQ,
				   E1432_ENABLE_ON));

	(void) sleep(1);
	break;
    case 24:
	/* Enable only freq data on last channel */
	if (verbose > 0)
	    (void) printf("Enable %s on chan %d\n",
			  fft ? "freq" : "time", nchan - 1);
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
		enabled[type][chan] = 0;
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_OFF));
	CHECK(e1432_set_enable(hw, group,
			       E1432_ENABLE_TYPE_FREQ,
			       E1432_ENABLE_OFF));
	CHECK(e1432_set_enable(hw, chan_list[nchan - 1],
			       fft ? E1432_ENABLE_TYPE_FREQ :
			       E1432_ENABLE_TYPE_TIME,
			       E1432_ENABLE_ON));
	enabled[fft ? 1 : 0][nchan - 1] = 1;

	(void) sleep(1);
	break;
    }

    return 0;
}

static int
change_enable(E1432ID hw, int group, int nchan,
	      SHORTSIZ16 *chan_list, int fft, int mode,
	      int enabled[][NCHAN_MAX], int verbose)
{
    switch (mode)
    {
    case MODE_MISC:
	return change_enable_misc(hw, group, nchan, chan_list,
				  fft, enabled);
    case MODE_ONOFF:
	return change_enable_onoff(hw, group, nchan, chan_list,
				   fft, enabled);
    case MODE_SCAN:
	return change_enable_scan(hw, group, nchan, chan_list,
				  fft, enabled);
    case MODE_SLEEP:
	return change_enable_sleep(hw, group, nchan, chan_list,
				   fft, enabled, verbose);
    }
    return 0;
}

static void
error(int nchan, int enabled[][NCHAN_MAX], int scan)
{
    int     chan;

    (void) fprintf(stderr, "  Scan: %d\n", scan);
    (void) fprintf(stderr, "  Enabled Time Data:");
    for (chan = 0; chan < nchan; chan++)
	if (enabled[0][chan])
	    (void) fprintf(stderr, " %d", chan);
    (void) fprintf(stderr, "\n  Enabled Freq Data:");
    for (chan = 0; chan < nchan; chan++)
	if (enabled[1][chan])
	    (void) fprintf(stderr, " %d", chan);
    (void) fprintf(stderr, "\n");
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type)
{
    double  tmp;
    int     df2, df5;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (trailer->rpm1 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm1 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm1,
		       *(long *) &trailer->rpm1);
	return -1;
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    if (df2 != ((trailer->info & E1432_TRAILER_INFO_DEC_2_MASK)
		>> E1432_TRAILER_INFO_DEC_2_SHIFT))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    timeout = (2 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    CHECK(e1432_print_errors(0));
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    CHECK(e1432_print_errors(1));
	    return -1;
	}
    CHECK(e1432_print_errors(1));
    if (status < 0)
    {
	if (status == ERR1432_FIFO_OVERRUN)
	{
	    if (verbose > 0)
		(void) printf("Fifo overflow\n");
	}
	else
	    (void) fprintf(stderr,
			   "%s: e1432_block_available: returned %d\n",
			   progname, status);
	/* Return status, not -1, so we can notice fifo overflow in
	   the caller. */
	return status;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

/*ARGSUSED*/
static int
run(E1432ID hw, int group, int nmod, int nchan, SHORTSIZ16 *chan_list,
    int data_mode, int fft, int mode, int nscan, int verbose)
{
    FLOATSIZ64 buffer[BLOCKSIZE];
    LONGSIZ32 count, expect;
    struct e1432_trailer trailer;
    double  span;
    int     enabled[2][NCHAN_MAX];
    int     scan, type, chan, status;

    if (data_mode == E1432_CONTINUOUS_MODE)
	span = 4000;
    else
	span = 20000;

    /* Initialize enabled array */
    for (type = 0; type < 2; type++)
	for (chan = 0; chan < nchan; chan++)
	    if (type > 0 && !fft)
		enabled[type][chan] = 0;
	    else
		enabled[type][chan] = 1;

    for (scan = 0; scan < nscan; scan++)
    {
	/* Maybe change the enable settings.  For multi-module
	   measurements, we can't reliably change the enable setting
	   after calling e1432_block_available, so do it before
	   instead. */
	if (nmod > 1)
	    if (change_enable(hw, group, nchan, chan_list,
			      fft, mode, enabled, verbose) < 0)
	    {
		error(nchan, enabled, scan);
		return -1;
	    }

	/* Wait for block available */
	status = wait_block_avail(hw, group, scan, verbose,
				  BLOCKSIZE, span);
	if (status == ERR1432_FIFO_OVERRUN)
	    /* Not an error, just exit the measurement */
	    break;
	if (status < 0)
	{
	    error(nchan, enabled, scan);
	    return -1;
	}

	/* Maybe change the enable settings.  For single-module
	   measurements, change the enable settings after calling
	   e1432_block_available */
	if (nmod <= 1)
	    if (change_enable(hw, group, nchan, chan_list,
			      fft, mode, enabled, verbose) < 0)
	    {
		error(nchan, enabled, scan);
		return -1;
	    }

	/* Changing enable settings could have momentarily enabled
	   nothing, during which time an overflowed fifo could have
	   emptied, resulting in no data now.  Check for that. */
	CHECK(e1432_print_errors(0));
	status = e1432_block_available(hw, group);
	CHECK(e1432_print_errors(1));
	if (status == ERR1432_FIFO_OVERRUN)
	    /* Not an error, just exit the measurement */
	    break;
	if (status < 0)
	{
	    (void) fprintf(stderr,
			   "%s: e1432_block_available: returned %d\n",
			   progname, status);
	    error(nchan, enabled, scan);
	    return -1;
	}

	/* Read the data */
	for (type = 0; type < 2; type++)
	    for (chan = 0; chan < nchan; chan++)
	    {
		if (!enabled[type][chan])
		    continue;

		if (verbose > 1)
		    (void) printf("Reading chan %d type %d\n",
				  chan, type);

		expect = (type == 0 ? BLOCKSIZE : FBLOCKSIZE);
		if (e1432_read_float64_data(hw, chan_list[chan],
					    type == 0 ?
					    E1432_TIME_DATA :
					    E1432_FREQ_DATA,
					    buffer, expect,
					    &trailer,
					    &count) < 0)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_float64_data: "
				   "returned negative\n",
				   progname);
		    error(nchan, enabled, scan);
		    return -1;
		}
		if (count != expect)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_float64_data: "
				   "actual count was %ld\n",
				   progname, count);
		    error(nchan, enabled, scan);
		    return -1;
		}

		if (check_trailer(&trailer, 51200, span,
				  chan_list[chan], type) < 0)
		{
		    error(nchan, enabled, scan);
		    return -1;
		}
	    }

	/* Maybe change the enable settings */
	if (change_enable(hw, group, nchan, chan_list,
			  fft, mode, enabled, verbose) < 0)
	{
	    error(nchan, enabled, scan);
	    return -1;
	}
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-bcfuvV] [-L laddr] [-m mode] [-n nchan]\n"
		   "\t[-N nmod] [-s scans]\n"
		   "\t-b: Use block mode\n"
		   "\t-c: Use continuous mode, and lower span\n"
		   "\t-f: Set up for fft data\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Use <mode>, default mode 0\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-s: Run for <scans> scans, default 1000\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    int     c, i, data_mode, fft, mode, nmod, nscan, verbose;
    int     group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    data_mode = E1432_DATA_MODE_OVERLAP_BLOCK;
    fft = 0;
    laddr[0] = 8;
    mode = 0;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    nscan = 1000;
    verbose = 0;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "bcfL:m:n:N:s:uvV")) != -1)
	switch (c)
	{
	case 'b':
	    data_mode = E1432_BLOCK_MODE;
	    break;
	case 'c':
	    data_mode = E1432_CONTINUOUS_MODE;
	    break;
	case 'f':
	    fft = 1;
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    mode = strtol(optarg, &p, 0);
	    if (optarg == p || mode < 0 || mode > MODE_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid mode: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 's':
	    nscan = strtol(optarg, &p, 0);
	    if (optarg == p || nscan < 0)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of scans: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, data_mode, fft) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nmod, nchan, chan_list,
	    data_mode, fft, mode, nscan, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
